implementation module StdPicture


//	Version 1.0

//	Drawing functions and other operations on Picture


import	StdInt, StdBool, StdReal, StdList, StdFunc
import	ospicture, osfont, osrgn, ostoolbox
import	commondef, StdPictureDef, StdFont


/*	Attribute functions.
*/
//	Get a shareable copy of the current picture.
getPicture :: !*Picture -> (!Picture,!*Picture)
getPicture picture
	= sharePicture picture

getPictureAttributes :: !Picture -> [PictureAttribute]
getPictureAttributes picture
	= getpictattributes picture

setPictureAttributes:: ![PictureAttribute] !*Picture -> *Picture
setPictureAttributes atts picture
	= setpictattributes atts picture

//	Pen position attributes:
setPenPos :: !Point !*Picture -> *Picture
setPenPos pos picture
	= setpictpenpos pos picture

getPenPos :: !Picture -> Point
getPenPos picture
	= getpictpenpos picture

class movePenPos figure	:: !figure !*Picture -> *Picture
//	Move the pen position as much as when drawing the figure.

instance movePenPos Vector where
	movePenPos :: !Vector !*Picture -> *Picture
	movePenPos v picture
		= movepictpenpos v picture

instance movePenPos Curve where
	movePenPos :: !Curve !*Picture -> *Picture
	movePenPos curve picture
		# (curpos,picture)	= accpictpenpos picture
		  (_,_,endpos)		= getcurve_rect_begin_end curpos curve
		# picture			= setpictpenpos endpos picture
		= picture


//	PenSize attributes:
setPenSize :: !Int !*Picture -> *Picture
setPenSize w picture
	= setpictpensize w picture

getPenSize :: !Picture -> Int
getPenSize picture
	= getpictpensize picture

setDefaultPenSize :: !*Picture -> *Picture
setDefaultPenSize picture
	= setpictpensize 1 picture


//	Colour attributes:
setPenColour :: !Colour !*Picture -> *Picture
setPenColour c picture
	= setpictpencolour c picture

getPenColour :: !Picture -> Colour
getPenColour picture
	= getpictpencolour picture

setDefaultPenColour :: !*Picture -> *Picture
setDefaultPenColour picture
	= setpictpencolour Black picture


//	Font attributes:
setPenFont :: !Font !*Picture -> *Picture
setPenFont f picture
	= setpictpenfont f picture

getPenFont :: !Picture -> Font
getPenFont picture
	= getpictpenfont picture

setDefaultPenFont :: !*Picture -> *Picture
setDefaultPenFont picture
	= setpictpendefaultfont picture


/*	Picture is an environment instance of the FontEnv class.	*/
instance FontEnv Picture where
	openFont :: !FontDef !*Picture -> (!(!Bool,!Font),!*Picture)
	openFont fontDef picture
		# (origin,pen,context,tb)	= unpackPicture picture
		# (found,font,tb)			= OSselectfont fontDef tb
		# picture					= packPicture origin pen context tb
		= ((found,font),picture)
	
	openDefaultFont :: !*Picture -> (!Font,!*Picture)
	openDefaultFont picture
		= accpicttoolbox OSdefaultfont picture
	
	openDialogFont :: !*Picture -> (!Font,!*Picture)
	openDialogFont picture
		= accpicttoolbox OSdialogfont picture
	
	getFontNames :: !*Picture -> (![FontName],!*Picture)
	getFontNames picture
		= accpicttoolbox OSfontnames picture
	
	getFontStyles :: !FontName	!*Picture -> (![FontStyle],!*Picture)
	getFontStyles fName picture
		= accpicttoolbox (OSfontstyles fName) picture
	
	getFontSizes :: !Int !Int !FontName	!*Picture -> (![FontSize],!*Picture)
	getFontSizes sizeBound1 sizeBound2 fName picture
		= accpicttoolbox (OSfontsizes sizeBound1 sizeBound2 fName) picture
	
	getFontCharWidth :: !Char !Font !*Picture -> (!Int,!*Picture)
	getFontCharWidth char font picture
		# (osPictContext,picture)	= peekOSPictContext picture
		# (widths,picture)			= accpicttoolbox (OSgetfontcharwidths True osPictContext [char] font) picture
		= (hd widths,picture)
	
	getFontCharWidths :: ![Char] !Font !*Picture -> (![Int],!*Picture)
	getFontCharWidths chars font picture
		# (osPictContext,picture)	= peekOSPictContext picture
		= accpicttoolbox (OSgetfontcharwidths True osPictContext chars font) picture
	
	getFontStringWidth :: !String !Font !*Picture -> (!Int,!*Picture)
	getFontStringWidth string font picture
		# (osPictContext,picture)	= peekOSPictContext picture
		# (widths,picture)			= accpicttoolbox (OSgetfontstringwidths True osPictContext [string] font) picture
		= (hd widths,picture)
	
	getFontStringWidths :: ![String] !Font !*Picture -> (![Int],!*Picture)
	getFontStringWidths strings font picture
		# (osPictContext,picture)	= peekOSPictContext picture
		= accpicttoolbox (OSgetfontstringwidths True osPictContext strings font) picture
	
	getFontMetrics :: !Font !*Picture -> (!FontMetrics,!*Picture)
	getFontMetrics font picture
		# (osPictContext,picture)						= peekOSPictContext picture
		# ((ascent,descent,leading,maxwidth),picture)	= accpicttoolbox (OSgetfontmetrics True osPictContext font) picture
		= ({fAscent=ascent,fDescent=descent,fLeading=leading,fMaxWidth=maxwidth},picture)


/*	Drawing functions.
	These functions are divided into the following classes:
		Drawables:	draw     'line-oriented' figures at the current  pen position.
					drawAt   'line-oriented' figures at the argument pen position.
		Fillables:	fill     'area-oriented' figures at the current  pen position.
					fillAt   'area-oriented' figures at the argument pen position.
		Hilites:	hilite	 draws figures in the appropriate 'hilite' mode at the current pen position.
					hiliteAt draws figures in the appropriate 'hilite' mode at the current pen position.
					Both functions reset the 'hilite' after drawing.
		Clips:		apply a list of drawing functions within a clipping area(s).
					clip     takes the base point of the area to be the current  pen position.
					clipAt   takes the base point of the area to be the argument pen position.
*/
class Drawables figure
where
	draw	::			!figure					!*Picture -> *Picture
	drawAt	:: !Point	!figure					!*Picture -> *Picture

class Fillables figure
where
	fill	::			!figure					!*Picture -> *Picture
	fillAt	:: !Point	!figure					!*Picture -> *Picture

class Hilites figure
where
	hilite	::			!figure					!*Picture -> *Picture
	hiliteAt:: !Point	!figure					!*Picture -> *Picture

class Clips area
where
	clip	::			!area ![DrawFunction]	!*Picture -> *Picture
	clipAt	:: !Point	!area ![DrawFunction]	!*Picture -> *Picture

::	DrawFunction
	:==	*Picture -> *Picture


/*	(draw/xor)Picture applies the given drawing functions to the given picture in left-to-right order.
	When drawing is done, all picture attributes are set to the attribute values of the original picture.
*/
drawPicture :: ![DrawFunction] !*Picture -> *Picture
drawPicture drawfs picture
	| isEmpty drawfs
	= picture
	# (spicture,picture)= getPicture picture
	  atts				= getPictureAttributes spicture
	# picture			= StrictSeq drawfs picture
	# picture			= setPictureAttributes atts picture
	= picture

xorPicture :: ![DrawFunction] !*Picture -> *Picture
xorPicture drawfs picture
	| isEmpty drawfs
	= picture
	# (spicture,picture)= getPicture picture
	  atts				= getPictureAttributes spicture
	# picture			= setpictxormode picture
	# picture			= StrictSeq drawfs picture
	# picture			= setpictnormalmode picture
	# picture			= setPictureAttributes atts picture
	= picture


/*	(draw/xor)seqPicture applies the given drawing functions to the given picture in left-to-right order.
	Picture attributes are not reset after drawing.
*/
drawseqPicture :: ![DrawFunction] !*Picture -> *Picture
drawseqPicture drawfs picture
	= StrictSeq drawfs picture

xorseqPicture :: ![DrawFunction] !*Picture -> *Picture
xorseqPicture drawfs picture
	# picture	= setpictxormode picture
	# picture	= StrictSeq drawfs picture
	# picture	= setpictnormalmode picture
	= picture


/*	Hiliting figures:
*/
instance Hilites Box
where
	hilite :: !Box !*Picture -> *Picture
	hilite box picture
		# picture			= setpicthilitemode picture
		# (curpos,picture)	= accpictpenpos picture
		# picture			= pictfillrect (boxtorect curpos box) picture
		# picture			= setpictnormalmode picture
		= picture
	
	hiliteAt :: !Point !Box !*Picture -> *Picture
	hiliteAt base box picture
		# picture	= setpicthilitemode picture
		# picture	= pictfillrect (boxtorect base box) picture
		# picture	= setpictnormalmode picture
		= picture

instance Hilites Rectangle
where
	hilite :: !Rectangle !*Picture -> *Picture
	hilite rectangle picture
		# picture	= setpicthilitemode picture
		# picture	= pictfillrect (rectangletorect rectangle) picture
		# picture	= setpictnormalmode picture
		= picture
	
	hiliteAt :: !Point !Rectangle !*Picture -> *Picture
	hiliteAt _ rectangle picture
		# picture	= setpicthilitemode picture
		# picture	= pictfillrect (rectangletorect rectangle) picture
		# picture	= setpictnormalmode picture
		= picture


/*	Drawing within in a clipping area:
*/
instance Clips [figure] | Clips figure
where
	clip :: ![figure] ![DrawFunction] !*Picture -> *Picture | Clips figure
	clip area drawfs picture
		# (curpos,picture)	= accpictpenpos picture
		= clipAt curpos area drawfs picture
	
	clipAt :: !Point ![figure] ![DrawFunction] !*Picture -> *Picture | Clips figure
	clipAt point [area:areas] drawfs picture
		= clipAt point areas drawfs (clipAt point area drawfs picture)
	clipAt _ _ _ picture
		= picture

instance Clips Box
where
	clip :: !Box ![DrawFunction] !*Picture -> *Picture
	clip box drawFs picture
		# (curpos,picture)	= accpictpenpos picture
		= clipAt curpos box drawFs picture
	
	clipAt :: !Point !Box ![DrawFunction] !*Picture -> *Picture
	clipAt base=:{x,y} {box_w,box_h} drawFs picture
		| box_w==0 || box_h==0
		= picture
		# (origin,pen,context,tb)	= unpackPicture picture
		# (newClipRgn,tb)			= osnewrgn tb
		# (newClipRgn,tb)			= osrectrgn (l-origin.x,t-origin.y, r-origin.x,b-origin.y) newClipRgn tb
		# picture					= packPicture origin pen context tb
		# (newClipRgn,picture)		= pictsetcliprgn newClipRgn picture
		# picture					= StrictSeq drawFs picture
		# picture					= apppicttoolbox (osdisposergn newClipRgn) picture
		= picture
	where
		(l,r)						= minmax x (x+box_w)
		(t,b)						= minmax y (y+box_h)

instance Clips Rectangle
where
	clip :: !Rectangle ![DrawFunction] !*Picture -> *Picture
	clip rectangle drawfs picture
		# (base,box)	= rectangletoboxat rectangle
		= clipAt base box drawfs picture
	
	clipAt :: !Point !Rectangle ![DrawFunction] !*Picture -> *Picture
	clipAt _ rectangle drawfs picture
		# (base,box)	= rectangletoboxat rectangle
		= clipAt base box drawfs picture

rectangletoboxat :: !Rectangle -> (!Point,!Box)
rectangletoboxat {corner1,corner2={x,y}}
	= (corner1,{box_w=x-corner1.x,box_h=y-corner1.y})

instance Clips Polygon
where
	clip :: !Polygon ![DrawFunction] !*Picture -> *Picture
	clip polygon drawFs picture
		# (curPenPos,picture)	= accpictpenpos picture
		= clipAt curPenPos polygon drawFs picture
	
	clipAt :: !Point !Polygon ![DrawFunction] !*Picture -> *Picture
	clipAt {x,y} {polygon_shape} drawFs picture
		# (origin,pen,context,tb)	= unpackPicture picture
		# (newClipRgn,tb)			= osnewrgn tb
		# (newClipRgn,tb)			= ospolyrgn (x-origin.x,y-origin.y) (map (\{vx,vy}->(vx,vy)) polygon_shape) newClipRgn tb
		# picture					= packPicture origin pen context tb
		# (newClipRgn,picture)		= pictsetcliprgn newClipRgn picture
		# picture					= StrictSeq drawFs picture
		# picture					= apppicttoolbox (osdisposergn newClipRgn) picture
		= picture


drawPoint :: !*Picture -> *Picture
drawPoint picture
	# (curpos,picture)	= accpictpenpos picture
	# picture			= pictdrawpoint curpos picture
	# picture			= setpictpenpos {curpos & x=curpos.x+1} picture
	= picture

drawPointAt :: !Point !*Picture -> *Picture
drawPointAt point picture
	# (curpos,picture)	= accpictpenpos picture
	# picture			= pictdrawpoint point picture
	# picture			= setpictpenpos curpos picture
	= picture


/*	Point connecting drawing operations:
*/
drawLineTo :: !Point !*Picture -> *Picture
drawLineTo pos picture
	= pictdrawlineto pos picture

drawLine :: !Point !Point !*Picture -> *Picture
drawLine pos1 pos2 picture
	= pictdrawline pos1 pos2 picture


/*	Text drawing operations:
*/
instance Drawables Char
where
	draw :: !Char !*Picture -> *Picture
	draw char picture
		= pictdrawchar char picture
	
	drawAt :: !Point !Char !*Picture -> *Picture
	drawAt pos char picture
		# (curpos,picture)	= accpictpenpos picture
		# picture			= setpictpenpos pos picture
		# picture			= pictdrawchar char picture
		# picture			= setpictpenpos curpos picture
		= picture

instance Drawables {#Char}
where
	draw :: !{#Char} !*Picture -> *Picture
	draw string picture
		= pictdrawstring string picture
	
	drawAt :: !Point !{#Char} !*Picture -> *Picture
	drawAt pos string picture
		# (curpos,picture)	= accpictpenpos picture
		# picture			= setpictpenpos pos picture
		# picture			= pictdrawstring string picture
		# picture			= setpictpenpos curpos picture
		= picture


/*	Vector drawing operations:
*/
instance Drawables Vector
where
	draw :: !Vector !*Picture -> *Picture
	draw {vx,vy} picture
		# (curpos,picture)	= accpictpenpos picture
		  endpos			= {x=curpos.x+vx,y=curpos.y+vy}
		# picture			= pictdrawlineto endpos picture
		= picture
	
	drawAt :: !Point !Vector !*Picture -> *Picture
	drawAt pos=:{x,y} {vx,vy} picture
		= pictdrawline pos {x=x+vx,y=y+vy} picture


/*	Oval drawing operations:
*/
instance Drawables Oval
where
	draw :: !Oval !*Picture -> *Picture
	draw oval picture
		# (curpos,picture)	= accpictpenpos picture
		# picture			= pictdrawoval curpos oval picture
		= picture
	
	drawAt :: !Point !Oval !*Picture -> *Picture
	drawAt pos oval picture
		= pictdrawoval pos oval picture

instance Fillables Oval
where
	fill :: !Oval !*Picture -> *Picture
	fill oval picture
		# (curpos,picture)	= accpictpenpos picture
		# picture			= pictfilloval curpos oval picture
		= picture
	
	fillAt :: !Point !Oval !*Picture -> *Picture
	fillAt pos oval picture
		= pictfilloval pos oval picture


/*	Curve drawing operations:
*/
instance Drawables Curve
where
	draw :: !Curve !*Picture -> *Picture
	draw curve picture
		# (curpos,picture)	= accpictpenpos picture
		# picture			= pictdrawcurve True curpos curve picture
		= picture
	
	drawAt :: !Point !Curve !*Picture -> *Picture
	drawAt point curve picture
		= pictdrawcurve False point curve picture

instance Fillables Curve
where
	fill :: !Curve !*Picture -> *Picture
	fill curve picture
		# (curpos,picture)	= accpictpenpos picture
		# picture			= pictfillcurve True curpos curve picture
		= picture
	
	fillAt :: !Point !Curve !*Picture -> *Picture
	fillAt point curve picture
		= pictfillcurve False point curve picture


/*	Box drawing operations:
*/
instance Drawables Box
where
	draw :: !Box !*Picture -> *Picture
	draw box picture
		# (curpos,picture)	= accpictpenpos picture
		# picture			= pictdrawrect (boxtorect curpos box) picture
		= picture
	
	drawAt :: !Point !Box !*Picture -> *Picture
	drawAt point box picture
		= pictdrawrect (boxtorect point box) picture

instance Fillables Box
where
	fill :: !Box !*Picture -> *Picture
	fill box picture
		# (curpos,picture)	= accpictpenpos picture
		# picture			= pictfillrect (boxtorect curpos box) picture
		= picture
	
	fillAt :: !Point !Box !*Picture -> *Picture
	fillAt pos box picture
		= pictfillrect (boxtorect pos box) picture

boxtorect :: !Point !Box -> (!Int,!Int,!Int,!Int)
boxtorect {x,y} {box_w,box_h}
	= (l,t, r,b)
where
	(l,r) = minmax x (x+box_w)
	(t,b) = minmax y (y+box_h)


/*	Rectangle drawing operations:
*/
instance Drawables Rectangle
where
	draw :: !Rectangle !*Picture -> *Picture
	draw rectangle picture
		= pictdrawrect (rectangletorect rectangle) picture
	
	drawAt :: !Point !Rectangle !*Picture -> *Picture
	drawAt _ rectangle picture
		= pictdrawrect (rectangletorect rectangle) picture

instance Fillables Rectangle
where
	fill :: !Rectangle !*Picture -> *Picture
	fill rectangle picture
		= pictfillrect (rectangletorect rectangle) picture
	
	fillAt :: !Point !Rectangle !*Picture -> *Picture
	fillAt _ rectangle picture
		= pictfillrect (rectangletorect rectangle) picture

rectangletorect :: !Rectangle -> (!Int,!Int,!Int,!Int)
rectangletorect {corner1,corner2}
	= (l,t, r,b)
where
	(l,r) = minmax corner1.x corner2.x
	(t,b) = minmax corner1.y corner2.y


/*	Polygon drawing operations:
*/
instance Drawables Polygon
where
	draw :: !Polygon !*Picture -> *Picture
	draw polygon picture
		# (curpos,picture)	= accpictpenpos picture
		# picture			= pictdrawpolygon curpos polygon picture
		= picture
	
	drawAt :: !Point !Polygon !*Picture -> *Picture
	drawAt base polygon picture
		= pictdrawpolygon base polygon picture

instance Fillables Polygon
where
	fill :: !Polygon !*Picture -> *Picture
	fill polygon picture
		# (curpos,picture)	= accpictpenpos picture
		# picture			= pictfillpolygon curpos polygon picture
		= picture
	
	fillAt :: !Point !Polygon !*Picture -> *Picture
	fillAt base polygon picture
		= pictfillpolygon base polygon picture
